/*
 Test/Debug of Octave functionality (on E1433)
 */

#include <stdlib.h>             /* For exit */
#include <stdio.h>              /* For printf */
#include <string.h>             /* For strcmp, strcat */
#include <math.h>               /* For sqrt */
#include "e1432.h"

#define MAX_CHANS       E1432_INPUT_CHANS

#define BLOCKSIZE       1000
#define SAMPLE_FREQ     65536

#define MAX_FRAMESIZE   64

#define MAX_LONG        0x7fffffff

/* default rpm parameters */
#define RPM_LOW		600.0
#define RPM_HIGH	6000.0
#define RPM_INTERVAL	500.0
#define TRIGGER_MODE	E1432_AUTO_TRIGGER

/* Wrap this around all the many function calls which might fail */
#define DEBUG(s)        s
#ifdef  __lint
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        exit(1);\
    }\
} while (func)
#else
#define CHECK(func)     \
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
        DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
        exit(1);\
    }\
} while (0)
#endif


/* enums for command line parm entry */

SHORTSIZ16 octave_mode_enums[] =
{
    E1432_OCTAVE_MODE_FULL, E1432_OCTAVE_MODE_FULL,
    E1432_OCTAVE_MODE_THIRD, E1432_OCTAVE_MODE_THIRD
};
char *octave_mode_strs[] = {
    "O", "o",
    "F", "f",
    "T", "t",
    NULL
};

SHORTSIZ16 octave_avg_mode_enums[] =
{
    E1432_OCTAVE_AVG_MODE_LIN, E1432_OCTAVE_AVG_MODE_LIN,
    E1432_OCTAVE_AVG_MODE_EXP, E1432_OCTAVE_AVG_MODE_EXP
};
char *octave_avg_mode_strs[] =
{
    "L", "l",
    "E", "e",
    NULL
};

SHORTSIZ16 octave_hold_mode_enums[] =
{
    E1432_OCTAVE_HOLD_MODE_OFF, E1432_OCTAVE_HOLD_MODE_OFF,
    E1432_OCTAVE_HOLD_MODE_MIN, E1432_OCTAVE_HOLD_MODE_MIN,
    E1432_OCTAVE_HOLD_MODE_MIN, E1432_OCTAVE_HOLD_MODE_MIN,
    E1432_OCTAVE_HOLD_MODE_MAX, E1432_OCTAVE_HOLD_MODE_MAX
};
char *octave_hold_mode_strs[] =
{
    "O", "o",
    "M", "m",
    "N", "n",
    "X", "x",
    NULL
};


#define VALID_OPTS      "a:c:h:i:m:n:s:t:uCRS:T"

void
print_usage(char *name)
{
#   if E12432_OCTAVE_MODE_DEF == E1432_OCTAVE_MODE_FULL
        char md = 'F';
#   else
        char md = 'T';
#   endif

#   if E1432_OCTAVE_AVG_MODE_DEF == E1432_OCTAVE_AVG_MODE_LIN
        char ad = 'L';
#   else
        char ad = 'E';
#   endif

#   if E1432_OCTAVE_HOLD_MODE_DEF == E1432_OCTAVE_HOLD_MODE_MAX
        char hd = 'X';
#   endif
#   if E1432_OCTAVE_HOLD_MODE_DEF == E1432_OCTAVE_HOLD_MODE_MIN
        char hd = 'N';
#   endif
#   if E1432_OCTAVE_HOLD_MODE_DEF == E1432_OCTAVE_HOLD_MODE_OFF
        char hd = 'O';
#   endif

    (void) fprintf(stderr,"usage: %s [options]\n",name);
    (void) fprintf(stderr,"options:\n");
    (void) fprintf(stderr,"    -a octave_avg_mode [Lin, Exp]"
      " (default %c)\n", ad);
    (void) fprintf(stderr,"    -c octave_time_const"
      " (default %g)\n", E1432_OCTAVE_TIME_CONST_DEF);
    (void) fprintf(stderr,"    -h octave_hold_mode [maX, miN, Off]"
      " (default %c)\n", hd);
    (void) fprintf(stderr,"    -i octave_int_time"
      " (default %g)\n", E1432_OCTAVE_INT_TIME_DEF);
    (void) fprintf(stderr,"    -m octave_mode [Full, Third]"
      " (default %c)\n", md);
    (void) fprintf(stderr,"    -n loops (default loop forever)\n");
    (void) fprintf(stderr,"    -s octave_start_freq"
      " (default %g)\n", E1432_OCTAVE_START_FREQ_DEF);
    (void) fprintf(stderr,"    -t octave_time_step"
      " (default %g)\n", E1432_OCTAVE_TIME_STEP_DEF);
    (void) fprintf(stderr,"    -C read current octave data also\n");
    (void) fprintf(stderr,"    -R to do RPM triggering\n");
    (void) fprintf(stderr,"       Input 2 VPk sine to tach chans,"
      " sweep 5 Hz to 105 Hz in 30 seconds.\n");
    (void) fprintf(stderr,"    -S octave_stop_freq"
      " (default %g)\n", E1432_OCTAVE_STOP_FREQ_DEF);
    (void) fprintf(stderr,"    -T read time data also\n");
}


void
parm_err(char *arg, char *parm_name)
{
    (void) fprintf(stderr, "parameter error: %s \"%s\" not converted\n",
      parm_name, arg);
    exit(2);
}


FLOATSIZ32
get_float(char *arg, char *parm_name)
{
    FLOATSIZ32 ftmp;
    int rtn = sscanf(optarg, "%f", &ftmp);

    if ( rtn != 1 ) parm_err(arg, parm_name);
    return ftmp;
}


LONGSIZ32
get_long(char *arg, char *parm_name)
{
    return (LONGSIZ32)(get_float(arg, parm_name) + .5);
}


SHORTSIZ16
get_enum(char *arg, SHORTSIZ16 *enums, char **strs, char *parm_name)
{
    while ( *strs != NULL )
    {
        if ( strcmp(arg, *strs++) == 0 ) return *enums;
        enums++;
    }
    parm_err(arg, parm_name);
}


int
main(int argn, char **argv)
{
    /* command line vars */
    int opt;               /* really char returned */

    /* state vars */
    int read_time = 0;
    int read_current = 0;
    int print_current = 0;
    LONGSIZ32 loops = -1;

    /* boiler plate vars */
    SHORTSIZ16 modules = 1;
    SHORTSIZ16 laddr = 8;  /* default logical address */
    E1432ID hw;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 in_chans = 0;
    SHORTSIZ16 in_chan_list[MAX_CHANS];
    SHORTSIZ16 in_group;

    /* rpm trigger */
    int do_rpm = 0;
    SHORTSIZ16 tach0, tach1, tachs;
    SHORTSIZ16 tach_list[2];
    float rpm;

    /* general use */
    int i, j;
    SHORTSIZ16 status, error;
    LONGSIZ32 loop;

    /* general api vars */
    LONGSIZ32 blocksize = BLOCKSIZE;
    FLOATSIZ32 range = 1.0;

    /* time data */
    SHORTSIZ16 time_enable;
    FLOATSIZ64 time_data[BLOCKSIZE];

    /* octave data */
    FLOATSIZ64 octave_data[MAX_CHANS][MAX_FRAMESIZE];
    FLOATSIZ64 *oct_data[MAX_CHANS];
    LONGSIZ32 count;

    /* octave api vars */
    SHORTSIZ16 octave_mode = E1432_OCTAVE_MODE_DEF;
    SHORTSIZ16 octave_avg_mode = E1432_OCTAVE_AVG_MODE_DEF;
    SHORTSIZ16 octave_hold_mode = E1432_OCTAVE_HOLD_MODE_DEF;
    FLOATSIZ32 octave_start_freq = E1432_OCTAVE_START_FREQ_DEF;
    FLOATSIZ32 octave_stop_freq = E1432_OCTAVE_STOP_FREQ_DEF;
    FLOATSIZ32 octave_int_time = E1432_OCTAVE_INT_TIME_DEF;
    FLOATSIZ32 octave_time_const = E1432_OCTAVE_TIME_CONST_DEF;
    FLOATSIZ32 octave_time_step = E1432_OCTAVE_TIME_STEP_DEF;

    /* derived octave vars */
    LONGSIZ32 octave_blocksize;


    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);

    while ( ( opt = getopt(argn, argv, VALID_OPTS) ) != EOF )
    {
        switch (opt)
        {
        case 'a':
            octave_avg_mode = get_enum(optarg, octave_avg_mode_enums,
              octave_avg_mode_strs, "octave_avg_mode");
            break;
        case 'c':
            octave_time_const = get_float(optarg, "octave_time_const");
            break;
        case 'h':
            octave_hold_mode = get_enum(optarg, octave_hold_mode_enums,
              octave_hold_mode_strs, "octave_hold_mode");
            break;
        case 'i':
            octave_int_time = get_float(optarg, "octave_int_time");
            break;
        case 'm':
            octave_mode = get_enum(optarg, octave_mode_enums, octave_mode_strs,
              "octave_mode");
            break;
        case 'n':
            loops = get_long(optarg, "read data loops");
            break;
        case 's':
            octave_start_freq = get_float(optarg, "octave_start_freq");
            break;
        case 't':
            octave_time_step = get_float(optarg, "octave_time_step");
            break;
        case 'C':
            read_current = 1;
            break;
        case 'R':
            do_rpm = 1;
            break;
        case 'S':
            octave_stop_freq = get_float(optarg, "octave_stop_freq");
            break;
        case 'T':
            read_time = 1;
            break;
        default:
            print_usage(argv[0]);
            exit(2);
        }
    }

    print_current = read_current;

    time_enable = read_time ? E1432_ENABLE_ON : E1432_ENABLE_OFF;

    CHECK(e1432_assign_channel_numbers(modules, &laddr, &hw));

    CHECK(e1432_get_hwconfig(modules, &laddr, &hwconfig));

    if ( ! hwconfig.oct_present)
    {
        (void) fprintf(stderr, "Octave option not present at LA %d\n", laddr);
        exit(1);
    }
    in_chans += hwconfig.input_chans;

    /* set up in_chan_list, in_group */
    if ( in_chans > MAX_CHANS )
    {
        (void) fprintf(stderr,
          "Warning: %d channels found, only %d will be used\n",
          in_chans, MAX_CHANS);
        in_chans = MAX_CHANS;
    }
    /* Create channel group */
    for (i = 0; i < in_chans; i++)
    {
        in_chan_list[i] = E1432_INPUT_CHAN(i+1);
        oct_data[i] = &octave_data[i][0];;
    }
    in_group = e1432_create_channel_group(hw, in_chans, in_chan_list);
    if (in_group >= 0)
    {
        DEBUG((void) printf("e1432_create_channel_group in_group return %d\n",
                            in_group));
        exit(1);
    }

    /* Initialize general hardware things */
    CHECK(e1432_set_analog_input(hw, in_group, E1432_INPUT_MODE_VOLT,
      E1432_INPUT_HIGH_NORMAL, E1432_ANTI_ALIAS_ANALOG_ON,
      E1432_COUPLING_DC, range));
    CHECK(e1432_set_blocksize(hw, in_group, blocksize));
    CHECK(e1432_set_clock_freq(hw, in_group, SAMPLE_FREQ));

    if ( do_rpm )
    {
        /* rpm trig setup */
        tach0 = tach_list[0] = E1432_TACH_CHAN(1);
        tach1 = tach_list[1] = E1432_TACH_CHAN(2);
        tachs = e1432_create_channel_group(hw, 2, tach_list);
        if (tachs >= 0)
        {
            (void) printf("create_channel_group for tachs returned %d\n",
	      tachs);
            exit(1);
        }

        CHECK(e1432_set_data_mode(hw, in_group, E1432_DATA_MODE_OVERLAP_BLOCK));

        CHECK(e1432_set_trigger_channel(hw, tach0, E1432_CHANNEL_ON));
        CHECK(e1432_set_active(hw, tach1, E1432_CHANNEL_OFF));

        CHECK(e1432_set_trigger_level(hw, tach0, E1432_TRIGGER_LEVEL_LOWER,
	  0.0));
        CHECK(e1432_set_trigger_level(hw, tach0, E1432_TRIGGER_LEVEL_UPPER,
	  0.0));
        CHECK(e1432_set_arm_mode(hw, in_group, E1432_ARM_RPM_RUNUP)); 
        CHECK(e1432_set_pre_arm_mode(hw, tach0, E1432_AUTO_ARM)); 
        CHECK(e1432_set_arm_channel(hw, tach0, E1432_CHANNEL_ON)); 
        CHECK(e1432_set_rpm_low(hw, tach0, RPM_LOW)); 
        CHECK(e1432_set_rpm_high(hw, tach0, RPM_HIGH)); 
        CHECK(e1432_set_rpm_interval(hw, tach0, RPM_INTERVAL)); 
        CHECK(e1432_set_auto_trigger(hw, in_group, TRIGGER_MODE));
    }

    /* octave setup */
    CHECK(e1432_set_octave_meas(hw, in_group, E1432_OCTAVE_MEAS_ON));
    CHECK(e1432_set_octave_mode(hw, in_group, octave_mode));
    CHECK(e1432_set_octave_avg_mode(hw, in_group, octave_avg_mode));
    CHECK(e1432_set_octave_hold_mode(hw, in_group, octave_hold_mode));
    CHECK(e1432_set_octave_start_freq(hw, in_group, octave_start_freq));
    CHECK(e1432_set_octave_stop_freq(hw, in_group, octave_stop_freq));
    CHECK(e1432_set_octave_int_time(hw, in_group, octave_int_time));
    CHECK(e1432_set_octave_time_const(hw, in_group, octave_time_const));
    CHECK(e1432_set_octave_time_step(hw, in_group, octave_time_step));

    /* turn on OCTAVE data, possibly turn off TIME data */
    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_OCTAVE,
      E1432_ENABLE_ON));
    CHECK(e1432_set_enable(hw, in_group, E1432_ENABLE_TYPE_TIME, time_enable));


    /* Start measurement */
    CHECK(e1432_init_measure(hw, in_group));

    /* must be done after measurement starts, since computed at start */
    CHECK(e1432_get_octave_blocksize(hw, in_group, &octave_blocksize));

    for ( loop = 0; loop != loops; loop++ )
    {
        do  /* Wait for block available and check for errors and warnings  */
        {
            status = e1432_block_available(hw, in_group);
        } while ( status != 1 );
        if ( status < 0 )
        {
            (void) fprintf(stderr, "block_available() error %d, loop %d\n",
              status, loop);
            exit(1);
        }

        /* read Time data */
        if ( read_time )
        {
            for ( i = 0; i < in_chans; i++ )
            {
                CHECK(e1432_read_float64_data(hw, in_chan_list[i],
                  E1432_TIME_DATA, time_data, blocksize, NULL, &count));
                if ( i == 0 )
                {
                    printf("time[%d] = %lg, %lg, %lg, %lg,...\n",
                      in_chan_list[i],
                      time_data[0], time_data[1], time_data[2], time_data[3]);
                }
            }
        }

	if ( do_rpm )
	{
	    CHECK(e1432_get_data_rpm(hw, tach0, &rpm));
	    printf("\nRPM: %g\n", rpm);
	}

        /* read Octave data */
        for ( i = 0; i < in_chans; i++ )
        {
            CHECK(e1432_read_float64_data(hw, in_chan_list[i],
              E1432_OCTAVE_DATA, oct_data[i], octave_blocksize,
              NULL, &count));
            if ( i == 0 )  /* just print 1 channel */
            {
                for ( j = 0; j < count; j += 4 )
                {
                    printf("  %12.6g  %12.6g  %12.6g  %12.6g\n",
                      octave_data[i][j], octave_data[i][j+1],
                      octave_data[i][j+2], octave_data[i][j+3]);
                }
            }
        }

        if ( read_current )
        {
            CHECK(e1432_get_current_data(hw, in_group, E1432_OCTAVE_DATA,
              E1432_DATA_SIZE_FLOAT64, (void *)oct_data, &count));
            if ( print_current )
            {
                i = 0; /* just do first channel */
                printf("current[%d] = %g %g %g ... %g %g\n",
                  in_chan_list[i],
                  *oct_data[i],
                  *(oct_data[i]+1),
                  *(oct_data[i]+2),
                  *(oct_data[i]+count-2),
                  *(oct_data[i]+count-1));
            }
        }
        printf("\n"); /* separate scans */
    }

    return 0;
}
